/* Easy Drawer by Felix (updated on 9 Nov 2023) */
var easydrawerSettings = {
	speedOpen: 50,
	speedClose: 350,
};
/* Version 1: Click Function */
$(document).ready(function(){
	$('.easydrawer').click(function(event) {
		event.preventDefault();
		event.stopPropagation();
		easydrawer.open($(this).data('easydrawer-id'), this, easydrawerSettings.speedOpen);
	});
	$('.drawer__close, .drawer__overlay').click(function() {
		easydrawer.closeAll()
	});
});
/* Version 2: Function with Promise */
var easydrawer = {
	/**
	 * Opens the target drawer.
	 * Usage: easydrawer.open()
	 * Add .then(() => { functions to run after opening drawer })
	 *
	 * @param {string} target - The ID of the target drawer to be opened.
	 * @param {HTMLElement} triggerElem - The element that triggers the opening of the target drawer.
	 * @param {number} [speed=easydrawerSettings.speedOpen] - The speed at which the target drawer opens (default: easydrawerSettings.speedOpen).
	 * @return {Promise} A promise that resolves when the target element is fully opened.
	 */
	open(target, triggerElem, speed  = easydrawerSettings.speedOpen) {
		return new Promise((resolve, reject) => {
			const targetElement = document.getElementById(target);
			targetElement.classList.add('is-active');
			triggerElem.classList.add('easydrawer-open');
			document.documentElement.classList.add('easydrawer-enabled');
			setTimeout(() => {
				targetElement.classList.add('is-visible');
				resolve();
			}, speed);
		});
	},
	/**
	 * Closes the target drawer
	 * Usage: easydrawer.close()
	 * Add .then(() => { functions to run after closing drawer })
	 *
	 * @param {string} target - The ID of the target drawer to be closed.
	 * @param {HTMLElement} triggerElem - The element that triggers the closing of the target drawer.
	 * @param {number} [speed=easydrawerSettings.speedClose] - The speed at which the target drawer closes (default: easydrawerSettings.speedClose).
	 * @return {Promise} A promise that resolves when the target drawer is fully closed.
	 */
	close(target, triggerElem, speed = easydrawerSettings.speedClose) {
		return new Promise((resolve, reject) => {
			const targetElement = document.getElementById(target);
			targetElement.classList.remove('is-visible');
			setTimeout(() => {
				targetElement.classList.remove('is-active');
				document.documentElement.classList.remove('easydrawer-enabled');
				triggerElem.classList.remove('easydrawer-open');
				resolve();
			}, speed);
		});
	},
	/**
	 * Closes all the drawers.
	 * Usage: easydrawer.closeAll()
	 * Add .then(() => { functions to run after closing drawers })
	 *
	 * @param {number} [speed=easydrawerSettings.speedClose] - The speed at which the drawers close. Defaults to the value set in easydrawerSettings.
	 * @return {Promise} A promise that resolves when all drawers are fully closed.
	 */
	closeAll(speed = easydrawerSettings.speedClose) {
		return new Promise((resolve, reject) => {
			const triggerElems = document.querySelectorAll('.easydrawer-open');
			const drawerElems = document.querySelectorAll('.is-active');
			drawerElems.forEach((drawerElem) => {
				drawerElem.classList.remove('is-visible');
			});
			setTimeout(() => {
				document.documentElement.classList.remove('easydrawer-enabled');
				triggerElems.forEach((triggerElem) => {
					triggerElem.classList.remove('easydrawer-open');
				});
				drawerElems.forEach((drawerElem) => {
					drawerElem.classList.remove('is-active');
				});
				resolve();
			}, speed);
		});
	}
}